/******************************************************************************
	[DDScreen.c]

	Implement ScreenInterface using Direct Draw.

	Copyright (C) 2004 Ki

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
******************************************************************************/
#define DIRECTDRAW_VERSION 0x0300

#include <ddraw.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>

#include "WinMain.h"
#include "ScreenInterface.h"
#include "DDScreen.h"
#include "MmxImpl.h"
#include "Printf.h"


/*--------------------------------------------------------------------------*
 *                           Static variables                               *
 *--------------------------------------------------------------------------*/
static LPDIRECTDRAW				_pDD         = NULL;  // DirectDraw object
static LPDIRECTDRAWSURFACE		_pDDSPrimary = NULL;  // DirectDraw primary surface
static LPDIRECTDRAWSURFACE		_pDDSBack    = NULL;  // DirectDraw offscreen surface

static LONG						_SurfacePitch = 0;

static int						_Width;        // width of display
static int						_Height;       // height of display
static Uint16*					_pPixels;

static Uint32					_MinimumTimerResolution;

static Uint32					_VirtualTicks;
static Uint32					_VirtualTicksDecimal;

static float					_FPS = 60.0;
static BOOL						_bSkipDraw;


static Sint32					_BitsPerPixel;
static Uint32					_Rmask;
static Uint32					_Gmask;
static Uint32					_Bmask;

static Uint32					_Rshift;
static Uint32					_Gshift;
static Uint32					_Bshift;

static Uint32					_Flags;
static BOOL						_bUpdate;

static BOOL						_bLocked = FALSE;


/*-----------------------------------------------------------------------------
	[Deinit]
		Finished with all objects we use; release them
-----------------------------------------------------------------------------*/
void
DDSCREEN_Deinit()
{
	if (_pDD != NULL)
	{
		if (_pDDSBack != NULL)
		{
			_pDDSBack->lpVtbl->Release(_pDDSBack);
			_pDDSBack = NULL;
		}

		if (_pDDSPrimary != NULL)
		{
			_pDDSPrimary->lpVtbl->Release(_pDDSPrimary);
			_pDDSPrimary = NULL;
		}

		_pDD->lpVtbl->Release(_pDD);
		_pDD = NULL;

		_Width  = 0;
		_Height = 0;

		if (timeEndPeriod(_MinimumTimerResolution) != TIMERR_NOERROR)
		{
			// REPORT WARNING
		}
	}
}


static
Uint32
get_shift(
	Uint32		mask)
{
	Uint32		i;

	for (i = 0; i < 32; i++)
	{
		if (mask & (1 << i))
		{
			return i;
		}
	}

	return 0;
}


/*-----------------------------------------------------------------------------
	[Init]
		XN[܂B 
-----------------------------------------------------------------------------*/
BOOL
DDSCREEN_Init(
	Sint32		width,
	Sint32		height,
	Sint32		bitsPerPixel,
	Uint32		flags)
{
	TIMECAPS		tc;
	HRESULT			hRet;
	DDSURFACEDESC	ddsd;
	HWND			hWnd = WINMAIN_GetHwnd();

	hRet = DirectDrawCreate(NULL, &_pDD, NULL);
	if (hRet != DD_OK)
		return FALSE;

	// xݒ肷B 
	hRet = _pDD->lpVtbl->SetCooperativeLevel(_pDD, hWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN);
//	hRet = _pDD->lpVtbl->SetCooperativeLevel(_pDD, hWnd, DDSCL_NORMAL);
	if (hRet != DD_OK)
	{
		PRINTF("SetCooperativeLevel() failed.");
		return FALSE;
	}

	// ʂ̉𑜓xݒ肷B
	hRet = _pDD->lpVtbl->SetDisplayMode(_pDD, width, height, 16);
	if (hRet != DD_OK)
	{
		PRINTF("SetDisplayMode() failed.");
		DDSCREEN_Deinit();
		return FALSE;
	}

	// vC}T[tFX쐬 
	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS;
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_VIDEOMEMORY;

	hRet = _pDD->lpVtbl->CreateSurface(_pDD, &ddsd, &_pDDSPrimary, NULL);
	if (hRet != DD_OK)
	{
		PRINTF("CreateSurface() failed.");
		DDSCREEN_Deinit();
		return FALSE;
	}

    // ItXN[T[tFX쐬 
	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY;
	ddsd.dwWidth = width;
	ddsd.dwHeight = height;

	hRet = _pDD->lpVtbl->CreateSurface(_pDD, &ddsd, &_pDDSBack, NULL);
	if (hRet != DD_OK)
	{
		PRINTF("CreateSurface(offscreen surface) failed.");
		DDSCREEN_Deinit();
		return FALSE;
	}

	// Get pixel format 
	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_PIXELFORMAT|DDSD_CAPS;
	_pDDSPrimary->lpVtbl->GetSurfaceDesc(_pDDSPrimary, &ddsd);

	_BitsPerPixel = ddsd.ddpfPixelFormat.dwRGBBitCount;
	_Rmask = ddsd.ddpfPixelFormat.dwRBitMask;
	_Gmask = ddsd.ddpfPixelFormat.dwGBitMask;
	_Bmask = ddsd.ddpfPixelFormat.dwBBitMask;

	// Rshift, Gshift, Bshift ߂ 
	_Rshift = get_shift(_Rmask);
	_Gshift = get_shift(_Gmask);
	_Bshift = get_shift(_Bmask);

	if (DDSCREEN_Lock())
	{
		DDSCREEN_FillRect(0, 0, width, height, 0);
		DDSCREEN_Unlock();
	}

	// Save our screen resolution
	_Width = width;
	_Height = height;
	_Flags = flags;

	if (timeGetDevCaps(&tc, sizeof(tc)) != TIMERR_NOERROR)
	{
		// REPORT WARNING
	}

	_MinimumTimerResolution = tc.wPeriodMin;

	if (timeBeginPeriod(_MinimumTimerResolution) != TIMERR_NOERROR)
	{
		// REPORT WARNING
	}

	_VirtualTicks = timeGetTime();
	_VirtualTicksDecimal = 0;

	_bLocked = FALSE;

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[GetWidth]
	  XN[̉Ԃ܂B
-----------------------------------------------------------------------------*/
Sint32
DDSCREEN_GetWidth()
{
	return _Width;
}


/*-----------------------------------------------------------------------------
	[GetHeight]
	  XN[̍Ԃ܂B
-----------------------------------------------------------------------------*/
Sint32
DDSCREEN_GetHeight()
{
	return _Height;
}


/*-----------------------------------------------------------------------------
	[GetRshift]
	  ꂽOtBbN[hŁCԐFɉrbgVtg邩
	Ԃ܂D(Ⴆ16bppł11C15bppł10)
**---------------------------------------------------------------------------*/
Sint32
DDSCREEN_GetRshift()
{
	return _Rshift;
}


/*-----------------------------------------------------------------------------
	[GetGshift]
	  ꂽOtBbN[hŁCΐFɉrbgVtg邩
	Ԃ܂D(Ⴆ16bppł5C15bppł5)
**---------------------------------------------------------------------------*/
Sint32
DDSCREEN_GetGshift()
{
	return _Gshift;
}


/*-----------------------------------------------------------------------------
	[GetBshift]
	  ꂽOtBbN[hŁCFɉrbgVtg邩
	Ԃ܂D(Ⴆ16bppł0C15bppł0)
**---------------------------------------------------------------------------*/
Sint32
DDSCREEN_GetBshift()
{
	return _Bshift;
}


/*-----------------------------------------------------------------------------
	[GetRmask]
	  ꂽOtBbN[hŁCԐF}XNrbgԂ܂D
	Ԃ܂D(Ⴆ16bppł0xf800, 15bppł 0x7c00)
**---------------------------------------------------------------------------*/
Uint32
DDSCREEN_GetRmask()
{
	return _Rmask;
}


/*-----------------------------------------------------------------------------
	[GetGmask]
	  ꂽOtBbN[hŁCΐF}XNrbgԂ܂D
	Ԃ܂D(Ⴆ16bppł0x07e0, 15bppł 0x03e0)
**---------------------------------------------------------------------------*/
Uint32
DDSCREEN_GetGmask()
{
	return _Gmask;
}


/*-----------------------------------------------------------------------------
	[GetBmask]
	  ꂽOtBbN[hŁCF}XNrbgԂ܂D
	Ԃ܂D(Ⴆ16bppł0x001f, 15bppł 0x001f)
**---------------------------------------------------------------------------*/
Uint32
DDSCREEN_GetBmask()
{
	return _Bmask;
}


/*-----------------------------------------------------------------------------
	[GetBytesPerPixel]
	  PsNZɎgpoCgԂ܂D
**---------------------------------------------------------------------------*/
Uint32
DDSCREEN_GetBytesPerPixel()
{
	return 2;
}


/*-----------------------------------------------------------------------------
	[PutPixel]
	  PsNZ`܂D16bpp p
**---------------------------------------------------------------------------*/
void
DDSCREEN_PutPixel(
	Sint32		x,
	Sint32		y,
	Uint32		pixel)
{
//	if (x > 0 && y > 0 && x < _Width && y < _Height)
//		_pBuf[y * _Width + x] = (Uint16)pixel;
}


//-----------------------------------------------------------------------------
// Restore all lost objects
//-----------------------------------------------------------------------------
/*
static
BOOL
restore_all_objects(void)
{
    return _pDDSPrimary->lpVtbl->Restore(_pDDSPrimary) == DD_OK; // &&
//    		_pDDSBack->lpVtbl->Restore(_pDDSBack) == DD_OK;
}
*/


/*-----------------------------------------------------------------------------
	[IsFullScreen]
		XN[tXN[Ȃ TRUE, EChEȂ FALSE Ԃ܂B
-----------------------------------------------------------------------------*/
BOOL
DDSCREEN_IsFullScreen()
{
	return TRUE;
}


/*-----------------------------------------------------------------------------
	[ToggleFullScreen]
		XN[EChE^tXN[ɐ؂ւ܂D
-----------------------------------------------------------------------------*/
BOOL
DDSCREEN_ToggleFullScreen()
{
	return FALSE;
}


/*-----------------------------------------------------------------------------
	[ChangeMode]
		XN[[hύX܂B 
-----------------------------------------------------------------------------*/
BOOL
DDSCREEN_ChangeMode(
	Sint32		width,
	Sint32		height,
	Sint32		bpp,
	Uint32		flags)
{
	if (_bLocked)
		return FALSE;

	DDSCREEN_Deinit();
	return DDSCREEN_Init(width, height, bpp, flags);
}


/*-----------------------------------------------------------------------------
	[SetFPS]
		Frames Per Second ύX܂B 
-----------------------------------------------------------------------------*/
void
DDSCREEN_SetFPS(
	float		fps)
{
	_FPS = fps;
}


/*-----------------------------------------------------------------------------
	[WaitVBlank]
		AԂ҂܂B 
-----------------------------------------------------------------------------*/
BOOL
DDSCREEN_WaitVBlank(
	BOOL		bForceWait)
{
	Uint32	ticksNow;

	if (_bLocked)
		return FALSE;

	if (_Flags & SCREEN_FSYNCTO60HZFULLSCREEN)
	{
		// ̃[h̏ꍇ͂ő҂Aprimary surface  Blt 钼Oő҂ 
		//  bForceWait w肳ꂽꍇ͂ő҂ 
		if (bForceWait)
			_pDD->lpVtbl->WaitForVerticalBlank(_pDD, DDWAITVB_BLOCKBEGIN, NULL);

		return TRUE;
	}

	if (_FPS == 0.0) return TRUE;

	_bSkipDraw = FALSE;

	_VirtualTicksDecimal = (_VirtualTicksDecimal & 0xffff) + (Uint32)(65536.0 * 1000.0 * 1.0 / _FPS);
	_VirtualTicks += _VirtualTicksDecimal >> 16;

	ticksNow = timeGetTime();

	if (ticksNow < _VirtualTicks)
	{
		/*
			zǂ _VirtualTicks - ticksNow < 1/FPS * 1000 ƂȂĂꍇƁC
			_VirtualTicks = 0xffffffff ɋ߂lł ticksNow  wrap around
			Ă܂ĂꍇlDO҂ƌ҂́C
			_VirtualTicks - ticksNow <= 1/FPS * 1000 ǂŔfD
		*/
		if (_VirtualTicks - ticksNow <= (_VirtualTicksDecimal >> 16))
		{
			if (_VirtualTicks - ticksNow > 4) Sleep(_VirtualTicks - ticksNow - 3);
			while (timeGetTime() < _VirtualTicks);
		}
		return TRUE;
	}
	else
	{
		if (ticksNow - _VirtualTicks >= (_VirtualTicksDecimal >> 16) * 4)
		{
			/*
				҂Ԃ 4 {ȏオɌo߂Ăꍇ҂ɕԂD
				4 {ȏ̏ꍇ́uXLbvĂʁvƔfĂ邱ƂɂȂD
			*/
			_VirtualTicks = ticksNow;
			return TRUE;
		}
		else if (ticksNow - _VirtualTicks >= (_VirtualTicksDecimal >> 16) * 2)
		{
			/*
				҂Ԃ̂Q{ȏオɌo߂Ăꍇ
				FALSE ԂD
			*/
			_bSkipDraw = TRUE;
			return FALSE;
		}
		else
		{
			/*
				O񂩂̌oߎԂɑ҂Ԃ𒴂Ăꍇ͉҂ɕԂD
			*/
			return TRUE;
		}
	}

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[ShouldDraw]
		`sȂׂAsȂȂׂԂ܂B 
-----------------------------------------------------------------------------*/
BOOL
DDSCREEN_ShouldDraw()
{
	return (_bSkipDraw == FALSE);
}


/*-----------------------------------------------------------------------------
	[Lock]
		XN[bNKv΃bN܂B
-----------------------------------------------------------------------------*/
BOOL
DDSCREEN_Lock()
{
	HRESULT				hRet;
	DDSURFACEDESC		ddsd;
	int					trial = 10;

	if (_bLocked)
		return FALSE;

	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	while (trial--)
	{
		hRet = _pDDSPrimary->lpVtbl->Lock(_pDDSPrimary, NULL, &ddsd, DDLOCK_WAIT, NULL);
		if (hRet == DD_OK)
		{
			_pPixels = (Uint16*)ddsd.lpSurface;
			_SurfacePitch = ddsd.lPitch;
			_bLocked = TRUE;
			return TRUE;
		}
		else if (hRet == DDERR_SURFACELOST)
		{
			_pDDSPrimary->lpVtbl->Restore(_pDDSPrimary);
		}
    }

	return FALSE;
}


void*
DDSCREEN_GetBuffer()
{
	if (_bLocked)
		return (void*)_pPixels;

	return NULL;
}


const
Sint32
DDSCREEN_GetBufferPitch()
{
	if (_bLocked)
		return _SurfacePitch/2;

	return 0;
}


/*-----------------------------------------------------------------------------
	[Unlock]
		XN[̃bN܂B
-----------------------------------------------------------------------------*/
void
DDSCREEN_Unlock()
{
//	_pDDSBack->lpVtbl->Unlock(_pDDSBack, NULL);
	_pDDSPrimary->lpVtbl->Unlock(_pDDSPrimary, NULL);
	_pPixels = NULL;
	_SurfacePitch = 0;
	_bLocked = FALSE;
}


/*-----------------------------------------------------------------------------
	[DrawText]
		obNobt@݂܂B 
	ĂԑO DDSCREEN_Lock() ܂傤B
-----------------------------------------------------------------------------*/
void
DDSCREEN_DrawText(
	const Sint32	x,
	const Sint32	y,
	const Uint32	fgColor,
	const Uint32	bgColor,
	const BOOL		bOpaque,
	const char*		pText)
{
	;
}


/*-----------------------------------------------------------------------------
	[FillRect]
		obNobt@Ɏw̐F̋``܂B
	ĂԑO DDSCREEN_Lock() ܂傤B
-----------------------------------------------------------------------------*/
void
DDSCREEN_FillRect(
	Sint32		x,
	Sint32		y,
	Sint32		width,
	Sint32		height,
	Uint32		color)
{
	int		i;
	Uint32	bpp			= 2;
	Uint32	dstPitch	= _SurfacePitch;
	Uint8*	pDst8		= (Uint8*)_pPixels + y * dstPitch + bpp * x;

	if (_SurfacePitch == 0)
		return;

	if (_pPixels == NULL)
		return;

	for (i = 0; i < height; i++)
	{
		memset(pDst8, color, width*bpp);
		pDst8 += dstPitch;
	}
}



static
inline
void
zoom_line_2x(
	Uint16*		pDst,
	Uint16*		pSrc,
	Sint32		width)
{
#if defined(__GNUC__) && defined(USE_INLINE_ASM)
	MMX_ScaleLine2x(pDst, pSrc, width);
/*
	width /= 4;
	__asm__ (
		"movl		%0,    %%edx	\n\t"
		"movl		%1,    %%eax	\n\t"
		"l1_%=:						\n\t"
		"movq		(%%eax), %%mm0	\n\t"
		"movq		%%mm0, %%mm1	\n\t"
		"movq		%%mm0, %%mm2	\n\t"
		"punpcklwd	%%mm0, %%mm1	\n\t"
		"punpckhwd	%%mm0, %%mm2	\n\t"
		"movq		%%mm1, (%%edx)	\n\t"
		"movq		%%mm2, 8(%%edx)	\n\t"
		"addl		$16,   %%edx	\n\t"
		"addl		$8,    %%eax    \n\t"
		"decl		%2				\n\t"
		"jnz		l1_%=			\n\t"
		"emms						\n\t"
		: "=m" (pDst)
		: "m" (pSrc), "m" (width)
		: "%eax", "%edx"
	);
*/
#else
	Uint32*		pDst32 = (Uint32*)pDst;
	Uint32*		pSrc32 = (Uint32*)pSrc;
	Uint32		data;
	int			j;

	for (j = 0; j < width/2; ++j)
	{
		data = *pSrc32 & 0xffff;
		data |= data << 16;
		*pDst32++ = data;

		data = *pSrc32++ & 0xffff0000;
		data |= data >> 16;
		*pDst32++ = data;
	}
#endif
}


static
inline
void
zoom2x(
	Uint16*		pDst,			// ] IMAGE ւ̃|C^
	Uint16*		pSrc,			// ] IMAGE ւ̃|C^
	Sint32		srcW,			// ]̕
	Sint32		srcH,			// ]̍
	Sint32		dstPitch,
	Sint32		srcPitch)
{
	int		i;

	if (srcW == 0 || srcH == 0)
		return;

	if (_Flags & SCREEN_FSCANLINED)
	{
		for (i = 0; i < srcH; ++i)
		{
			zoom_line_2x(pDst + i*2 * dstPitch, pSrc + i * srcPitch, srcW);
		}
	}
	else
	{
		for (i = 0; i < srcH; ++i)
		{
			zoom_line_2x(&pDst[(i*2+0) * dstPitch], pSrc + i * srcPitch, srcW);
			zoom_line_2x(&pDst[(i*2+1) * dstPitch], pSrc + i * srcPitch, srcW);
		}
	}
}


/*-------------------------------------------------------------
** Pg^k֐ (Œ菬_)
**-----------------------------------------------------------*/
static
inline
void
zoom1d_fp(
	Uint16*		dest,		// ]̃|C^
	Uint16*		src,		// ]̃|C^
	Sint32		destW,		// ]̕ (W̔{Ŏw肷邱)
	Sint32		srcW)		// ]̕
{
	Uint32	R;
	Sint32	Is;

	R = (srcW << 16) / destW;

	Is = 0;

	while (destW > 0)
	{
		*dest++ = src[Is >> 16];	Is += R;
		*dest++ = src[Is >> 16];	Is += R;
		*dest++ = src[Is >> 16];	Is += R;
		*dest++ = src[Is >> 16];	Is += R;
		*dest++ = src[Is >> 16];	Is += R;
		*dest++ = src[Is >> 16];	Is += R;
		*dest++ = src[Is >> 16];	Is += R;
		*dest++ = src[Is >> 16];	Is += R;
		destW -= 8;
	}
}


/*-----------------------------------------------------------------------------
** ʏ̂Qg^k֐ (Œ菬_)
**  zoom1d 𗘗p
**---------------------------------------------------------------------------*/
static
void
zoom2d_fp(
	Uint16*		dest,			// ] IMAGE ւ̃|C^
	Uint16*		src,			// ] IMAGE ւ̃|C^
	Sint32		destW,			// ]̕
	Sint32		destH,			// ]̍
	Sint32		srcX,			// ]̊JnwW
	Sint32		srcY,			// ]̊JnxW
	Sint32		srcW,			// ]̕
	Sint32		srcH,			// ]̍
	Sint32		dstPitch,
	Sint32		srcPitch)
{
	Sint32	R;
	Sint32	Id;
	Sint32	Is;

	Is = Id = 0;

	if (_Flags & SCREEN_FSCANLINED)
	{
		R = (srcH << 17) / destH;		// (srcH << 16) * 2 / destH;

		while (Id < destH)
		{
			zoom1d_fp(&dest[Id * dstPitch],
					  &src[((Is >> 16) + srcY) * srcPitch + srcX],
					  destW,
					  srcW);
			Id += 2;
			Is += R;
		}
	}
	else
	{
		R = (srcH << 16) / destH;

		while (Id < destH)
		{
			zoom1d_fp(&dest[Id * dstPitch],
					  &src[((Is >> 16) + srcY) * srcPitch + srcX],
					  destW,
					  srcW);
			++ Id;
			Is += R;
		}
	}
}


static
BOOL
lock_offscreen_surface(
	LONG*				pPitch,
	Uint16**			ppPixels)
{
	HRESULT				hRet;
	DDSURFACEDESC		ddsd;
	int					trial = 10;

	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	while (trial--)
	{
		hRet = _pDDSBack->lpVtbl->Lock(_pDDSBack, NULL, &ddsd, DDLOCK_WAIT, NULL);
		if (hRet == DD_OK)
		{
			*ppPixels = (Uint16*)ddsd.lpSurface;
			*pPitch = ddsd.lPitch;
			return TRUE;
		}
		else if (hRet == DDERR_SURFACELOST)
		{
			_pDDSBack->lpVtbl->Restore(_pDDSBack);
		}
    }

	return FALSE;
}


/*
	n[hEFAɗXgb``惂[h
	VRAM  VRAM ]
	(oCjA⊮)
*/
static
void
zoom2d_dd(
	Uint16*		pSrc,			// ]̃|C^
	Sint32		dstW,			// ]̕
	Sint32		dstH,			// ]̍
	Sint32		srcW,			// ]̕
	Sint32		srcH,			// ]̍
	Sint32		dstPitch,		// in pixel
	Sint32		srcPitch)		// in pixel
{
	LONG		pitch;
	Uint16*		pPixels;
	RECT		rcDst;
	RECT		rcSrc;
	int			i;

	// ̃[hł̓vC}T[tFXɒڏ܂Ȃ̂ Lock  
	DDSCREEN_Unlock();

	/* ܂ offscreen surface ɏ */
	if (_Flags & SCREEN_FSCANLINED)
	{
		// [XLCf[^] 
		if (lock_offscreen_surface(&pitch, &pPixels))
		{
			for (i = 0; i < srcH; i++)
			{
				CopyMemory(pPixels, pSrc, srcW*sizeof(Uint16));
				pPixels += pitch/2*2;
				pSrc += srcPitch;
			}
			_pDDSBack->lpVtbl->Unlock(_pDDSBack, NULL);
		}

		// ʒu̒ 
		SetRect(&rcDst, 0, 0, dstW, dstH);
		SetRect(&rcSrc, 0, 0, srcW, srcH*2);
	}
	else
	{
		// stretched mode ݂̂gp (srcH != 2) 
		if (dstW >= srcW * 2 && dstH >= srcH * 2 && srcH != 2)
		{
			if (lock_offscreen_surface(&pitch, &pPixels))
			{
				// Q{g] 
				zoom2x(pPixels, pSrc, srcW, srcH, pitch/2, srcPitch);
			}
			_pDDSBack->lpVtbl->Unlock(_pDDSBack, NULL);
			srcW *= 2;
			srcH *= 2;
		}
		else
		{
			// {f[^] 
			if (lock_offscreen_surface(&pitch, &pPixels))
			{
				for (i = 0; i < srcH; i++)
				{
					CopyMemory(pPixels, pSrc, srcW*sizeof(Uint16));
					pPixels += pitch/2;
					pSrc += srcPitch;
				}
				_pDDSBack->lpVtbl->Unlock(_pDDSBack, NULL);
			}
		}

		// ʒu̒ 
		SetRect(&rcDst, 0, 0, dstW, dstH);
		SetRect(&rcSrc, 0, 0, srcW, srcH);
	}

	// VBL  ҂ 
	if (_Flags & SCREEN_FSYNCTO60HZFULLSCREEN)
		_pDD->lpVtbl->WaitForVerticalBlank(_pDD, DDWAITVB_BLOCKBEGIN, NULL);

	// vC}T[tFX֓] 
	if (_pDDSPrimary->lpVtbl->Blt(_pDDSPrimary, &rcDst, _pDDSBack, &rcSrc, DDBLT_ASYNC, NULL) == DDERR_SURFACELOST)
	{
		_pDDSPrimary->lpVtbl->Restore(_pDDSPrimary);
	}
}


/*-----------------------------------------------------------------------------
	[Blt]
		pSrc obNobt@։摜݂܂Bg^kA
	At@uh󂯕t܂(8bpp ȊO)B 
	ĂԑO DDSCREEN_Lock() ܂傤B
-----------------------------------------------------------------------------*/
void
DDSCREEN_Blt(
	const void*		pSrc,
	Sint32			srcX,
	Sint32			srcY,
	Sint32			dstX,
	Sint32			dstY,
	Sint32			srcW,
	Sint32			srcH,
	Sint32			dstW,
	Sint32			dstH,
	Sint32			srcPitch,
	Uint32			alpha)
{
	int		i;
	Uint32	dstPitch	= _SurfacePitch / 2;
	Uint16*	pSrc16		= (Uint16*)pSrc + srcY * srcPitch + srcX;
	Uint16*	pDst16		= _pPixels + dstY * dstPitch + dstX;

	if (_pPixels == NULL || pSrc == NULL)
		return;

	if (_bUpdate)
	{
		memset(_pPixels, 0, _Width*_Height*2);
		_bUpdate = FALSE;
	}

	if (srcW == dstW && srcH == dstH)	// {[h 
	{
		for (i = 0; i < dstH; i++)
		{
			memcpy(pDst16, pSrc16, dstW*2);
			pDst16 += dstPitch;
			pSrc16 += srcPitch;
		}
	}
	else if (srcW*2 == dstW && srcH*2 == dstH)	// 2 {g僂[h 
	{
		zoom2x(pDst16, pSrc16, srcW, srcH, dstPitch, srcPitch);
	}
	else
	{
		if (_Flags & SCREEN_FHARDWAREACCELERATION)
			zoom2d_dd(pSrc16, dstW, dstH, srcW, srcH, dstPitch, srcPitch);
		else
			zoom2d_fp(pDst16, pSrc16, dstW, dstH, 0, 0, srcW, srcH, dstPitch, srcPitch);
	}
}


/*-----------------------------------------------------------------------------
	[Update]
		obNobt@̓eʂɕ\܂B
	x, y, width, height Să[ɂƁAʂ̑S͈͂XV܂B
-----------------------------------------------------------------------------*/
void
DDSCREEN_Update(
	Sint32		x,
	Sint32		y,
	Sint32		width,
	Sint32		height)
{
/*
	static Sint32	_FrameCount = 0;
	static Sint32	_LastTick = 0;

	Uint32			ticksNow;
*/
	if (_bLocked)
		return;
/*
	if ((ticksNow = timeGetTime()) >= _LastTick + 10000)
	{
		printf("%lf [FPS]\n", (double)_FrameCount / ((double)(ticksNow - _LastTick) / 1000.0));
		_LastTick = ticksNow;
		_FrameCount = 0;
	}

	++_FrameCount;
*/
	if (x == 0 && y == 0 && width == 0 && height == 0)
	{
		_bUpdate = TRUE;
	}
}

